using System;
using System.Collections;
using System.DirectoryServices;
using System.Management;
using System.Net;
using System.Runtime.InteropServices;
using System.Security.Principal;
using System.Threading;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>3/4/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	///		Represents Windows user mapped to VBECS user. Contains utility methods for 
	///		peforming VBECS user checks agains Windows security.
	/// </summary>

	#endregion

	public class VbecsWindowsUser 
	{				
		#region variables

		/// <summary>
		/// VBECS Windows group name.
		/// </summary>

		// Used to define initial capacity of the generated users list. Not important - used to work arolund the MS bug.
		private const int ExpectedUserCount = 25;

		private const string ClassToStringTemplate = "{0}: LoginId={1}, FullName={2}";
		private const string LocalGroupNameTemplate = @"{0}\{1}";

		// WMI related 
		private const string UserAccountTypeId = "512";
		private const string UserAccountTypePropertyName = "AccountType";
		private const string WmiClassSystemPropertyName = "__CLASS";
		private const string UserAccountWmiClassName = "Win32_UserAccount";
		private const string UserAccountLoginIdPropertyName = "Name";
		private const string UserAccountFullNamePropertyName = "FullName";
		private const string WmiQueryTemplate = 
			@"	ASSOCIATORS OF {{Win32_Group.Domain='{0}',Name='{1}'}} 
				WHERE AssocClass=Win32_GroupUser Role=GroupComponent ResultRole=PartComponent";

		// Directory services related
		private const long GroupDoesNotExistComErrorCode = -2147022676;
		private const string AdsiGroupCheckQuery = "WinNT://.,computer";
		private const string AdsiGroupTypeName = "group";

		private readonly string _loginId;
		private readonly string _fullName;

		#endregion


		#region constructor

		/// <summary>
		/// Constructs an instance of the class from supplied parameters. 
		/// </summary>
		/// <param name="loginId">Windows user login ID.</param>
		/// <param name="fullName">Windows user full name.</param>
		private VbecsWindowsUser( string loginId, string fullName )
		{
			if( loginId == null )
				throw( new ArgumentNullException( "loginId" ) );

			_loginId = loginId.Trim().ToUpper();
			_fullName = ParseUserName(fullName);
		}

		#endregion


		#region properties

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6846"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Non-null, not empty, all caps string.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6847"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Short name (Windows login ID).
		/// </summary>
		public virtual string LoginId
		{
			get
			{
				return _loginId;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6848"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>No exception is thrown.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6849"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Full user name (person's real name). 
		/// </summary>
		public virtual string FullName
		{
			get
			{
				return _fullName;
			}
		}

		#endregion


		#region methods

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6850"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Non-null, non-emtpy string containing both login ID and full name.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6851"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Converts object to string representation by printing out class name and property values.
		/// </summary>
		/// <returns>String representation of the object.</returns>
		public override string ToString()
		{
			return String.Format( ClassToStringTemplate, this.GetType().Name, LoginId, FullName );
		}


		///<Developers>
		///	<Developer>Jason Engstrom</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/21/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8313"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8314"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Determines if current user is an Administrator
		/// </summary>
		/// <returns></returns>
		public static bool IsCurrentUserAdministrator()
		{
			Common.VbecsConfig config = Common.VbecsConfig.Current;

			return IsCurrentUserInDomainGroup(true);
		}


		///<Developers>
		///	<Developer>Jason Engstrom</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/21/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8311"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8312"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Determines if current user is a valid Vbecs User
		/// </summary>
		/// <returns></returns>
		public static bool IsCurrentUserVbecsUser()
		{
			Common.VbecsConfig config = Common.VbecsConfig.Current;

			return IsCurrentUserInDomainGroup(false);
		}


		/// <summary>
		/// Checks if current user executing the code is a member of the specified Active Directory Domain Group. 
		/// </summary>
		/// <returns>True if user is a member of the group, otherwise - false.</returns>
		private static bool IsCurrentUserInDomainGroup(bool admin)
		{   
			IList _cachedWindowsUsersList = VbecsWindowsUser.GetVbecsUsers(admin);
			
			foreach( VbecsWindowsUser _user in _cachedWindowsUsersList )
			{
				WindowsPrincipal principal = GetCurrentPrincipal();
				string userName = principal.Identity.Name;
				
				if (userName.Substring(userName.IndexOf("\\")+1).ToUpper() == _user.LoginId.ToUpper())
				{
					return true;
				}
			}

			// not found
			return false;
		}


		///<Developers>
		///	<Developer>Jason Engstrom</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6842"> 
		///		<ExpectedInput>VBECS AD Domain Group exists on the AD Server.</ExpectedInput>
		///		<ExpectedOutput>True.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6843"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks if VBECS AD Domain Group exists on the AD Server. 
		/// </summary>
		/// <returns>True if VBECS Active Directory Domain Group exists on the AD Server.</returns>
		public static bool DoesVbecsDomainGroupExist(bool adminGroup)
		{			
			var config = Common.VbecsConfig.Current;
            //CR 3554
		    using (var serverEntry = GetActiveDirectoryServerNode())
		    {

		        if (serverEntry != null)
		        {
		            using (var search = new DirectorySearcher {SearchRoot = serverEntry, SearchScope = SearchScope.Subtree})
		            {
		                if (adminGroup)
		                {
		                    search.Filter = "(&(objectClass=group)(cn=" + config.VbecsAdminDomainGroupName + "))";
		                }
		                else
		                {
		                    search.Filter = "(&(objectClass=group)(cn=" + config.VbecsUserDomainGroupName + "))";
		                }
		                var result = search.FindOne();

		                if (result != null)
		                {
		                    return true;
		                }
		            }
		            return false;
		        }
		        return false;
		    }
		}


		/// <summary>
		/// Gets current Windows principal representing logged on user. 
		/// </summary>
		/// <returns>Windows principal representing logged on user.</returns>
		private static WindowsPrincipal GetCurrentPrincipal()
		{
			return new WindowsPrincipal( WindowsIdentity.GetCurrent() );			
		}


		///<Developers>
		///	<Developer>Jason Engstrom</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6844"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Non-null, non-empty list of users.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6845"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves Windows users that are members of the AD Domain Group by executing query on 
		/// the AD server.
		/// </summary>
		/// <returns>List containing VBECS users.</returns>
		public static IList GetVbecsUsers(bool adminGroup)
		{				
			Common.VbecsConfig config = Common.VbecsConfig.Current;
			
            //CR 3554
		    using (var serverEntry = GetActiveDirectoryServerNode())
            using (var deSearch = new DirectorySearcher())
		    {
		            if (serverEntry == null) return null;
		            deSearch.SearchRoot = serverEntry;
		            deSearch.SearchScope = SearchScope.Subtree;

		            if (adminGroup)
		            {
		                deSearch.Filter = "(&(objectClass=group)(cn=" + config.VbecsAdminDomainGroupName + "))";
		            }
		            else
		            {
		                deSearch.Filter = "(&(objectClass=group)(cn=" + config.VbecsUserDomainGroupName + "))";
		            }

		            SearchResult result = deSearch.FindOne();

		        using (var deGroup = new DirectoryEntry(result.Path))
		        {
		            ArrayList resultList = new ArrayList(ExpectedUserCount);

		            //Get all members in the current Group node
		            object members = deGroup.Invoke("Members", null);

		            foreach (object member in (IEnumerable) members)
		            {
		                using (var dirMember = new DirectoryEntry(member))
		                using (var userSearch = new DirectorySearcher())
		                {
		                    //Get the User node for this member
		                    userSearch.SearchRoot = dirMember;
		                    userSearch.PropertiesToLoad.Add("samAccountName");

		                    SearchResult userResult = userSearch.FindOne();

		                    using (var deUser = new DirectoryEntry(userResult.Path))
		                    {

		                        string userName = deUser.Name.Remove(0, 3);
		                        int parsePoint = userName.IndexOf(@"\");

		                        if (parsePoint > -1)
		                        {
		                            // CR2366
		                            string firstName = userName.Remove(0, parsePoint + 2).Trim();
		                            // CR2366 end
		                            string lastName = userName.Remove(parsePoint, userName.Length - parsePoint);

		                            userName = firstName + " " + lastName;
		                        }

		                        string loginId = deUser.Properties["samAccountName"].Value.ToString();

		                        resultList.Add(new VbecsWindowsUser(loginId, userName));
		                    }
		                }
		            }
		            return resultList;
		        }
		    }
		}
		

		/// <summary>
		/// Retrieves a value of a string property from a WMI account object. 
		/// </summary>
		/// <param name="userAccount">WMI Win32 user account.</param>
		/// <param name="propertyName">Property name to get value for.</param>
		/// <returns>A value for the property with the given name.</returns>
		private static string GetWmiWin32AccountStringProperty( ManagementObject userAccount, string propertyName )
		{
			return userAccount.Properties[ propertyName ].Value.ToString();
		}


		/// <summary>
		/// Retrieves the Active Directory Server Node from the YURDOMAIN (aka vha.domain.ext) domain. 
		/// </summary>
		private static DirectoryEntry GetActiveDirectoryServerNode()
		{
			// Get domain name
			Common.VbecsConfig config = Common.VbecsConfig.Current;
			string domain = config.Domain;
			
			// Get directory entry
			return new DirectoryEntry("LDAP://" + domain);
		}


		/// <summary>
		/// Parses user name
		/// </summary>
		/// <param name="fullName"></param>
		/// <returns></returns>
		private string ParseUserName(string fullName)
		{
			//If there is no comma in the name, it is already correctly formatted
			if (fullName.IndexOf(",") > 0) 
			{
				string[] parsedName;
				char[] identifier = new char[] {','};

				//Split the user name at the comma
				parsedName = fullName.Split(identifier,fullName.Length);

				//Remove any fluff spaces (CR 2251)
				parsedName[0] = parsedName[0].Trim();
				parsedName[1] = parsedName[1].Trim();

				//Look for a parenthesis and get its location
				int parenLocation = parsedName[1].IndexOf("(");

				//Remove everything from the open paren through the closing paren
				if (parenLocation > -1)
				{
					parsedName[1] = parsedName[1].Remove(parenLocation-1,parsedName[1].Length - parenLocation+1);			
				}

				return parsedName[1] + " " + parsedName[0];
			}
			return fullName;
		}

		#endregion
	}
}